<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Models\Package;
use App\Models\Payment_history;
use App\Models\FileUploader;
use Illuminate\Support\Facades\DB;

class PackageController extends Controller
{
    public function index(Request $request)
    {
        $query = Package::query();

        if ($request->has('search') && $request->search != '') {
            $query->where('title', 'LIKE', '%' . $request->search . '%');
        }

        $packages = $query->orderBy('id', 'asc')->paginate(20);

        if ($request->has('search')) {
            $packages->appends(['search' => $request->search]);
        }

        return view('admin.package.index', compact('packages'));
    }


    public function store(Request $request)
    {
        $validated = $request->validate([
            'title'           => 'required|string|max:255|unique:packages,title',
            'description'     => 'nullable|string',
            'price'           => 'required|numeric|min:0',
            'interval'   => 'required|in:monthly,yearly,days',
            'interval_preiod'=> 'nullable|integer|min:1',
            'status'          => 'required|boolean',
            'features'        => 'nullable|string', // comma separated
        ]);

        // Convert comma-separated features into JSON array
        $features = [];
        if (!empty($validated['features'])) {
            $features = array_map('trim', explode(',', $validated['features']));
        }

        Package::create([
            'title'           => $validated['title'],
            'description'     => $validated['description'] ?? null,
            'price'           => $validated['price'],
            'interval'   => $validated['interval'],
            'interval_preiod'=> $validated['interval_preiod'] ?? null,
            'status'          => $validated['status'],
            'features'        => json_encode($features),
        ]);

        return redirect()->route('admin.package.index')
            ->with('success', get_phrase('Package added successfully'));
    }

    public function edit($id)
    {
        $page_data['package'] = Package::findOrFail($id);
        return view('admin.package.edit', $page_data);
    }


    public function update(Request $request, $id)
    {
        $package = Package::findOrFail($id);

        $validated = $request->validate([
            'title'           => 'required|string|max:255|unique:packages,title,' . $package->id,
            'description'     => 'nullable|string',
            'price'           => 'required|numeric|min:0',
            'interval'   => 'required|in:monthly,yearly,days',
            'interval_preiod'=> 'nullable|integer|min:1',
            'status'          => 'required|boolean',
            'features'        => 'nullable|string',
        ]);

        $features = [];
        if (!empty($validated['features'])) {
            $features = array_map('trim', explode(',', $validated['features']));
        }

        $package->update([
            'title'           => $validated['title'],
            'description'     => $validated['description'] ?? null,
            'price'           => $validated['price'],
            'interval'   => $validated['interval'],
            'interval_preiod'=> $validated['interval_preiod'] ?? null,
            'status'          => $validated['status'],
            'features'        => json_encode($features),
        ]);

        return redirect()->route('admin.package.index')
            ->with('success', get_phrase('Package updated successfully'));
    }

    public function delete($id)
    {
        $package = Package::findOrFail($id);

        $package->delete();

        return redirect()->route('admin.package.index')
            ->with('success', get_phrase('Package deleted successfully'));
    }

    public function payment_history_index(Request $request)
    {
        $packages = Package::all();

        $query = Payment_history::with(['company', 'package']);

        if ($request->has('search') && $request->search != '') {
            $search = $request->search;

            $query->where(function ($q) use ($search) {
                $q->whereHas('company', function ($q2) use ($search) {
                    $q2->where('name', 'LIKE', '%' . $search . '%');
                })->orWhereHas('package', function ($q2) use ($search) {
                    $q2->where('title', 'LIKE', '%' . $search . '%');
                });
            });
        }

        if ($request->has('status') && $request->status != 'all') {
            $status = $request->status;
            $query->whereHas('company', function ($q) use ($status) {
                $q->where('status', $status);
            });
        }

        if ($request->has('package_id') && $request->package_id != 'all') {
            $packageId = $request->package_id;
            $query->whereHas('package', function ($q) use ($packageId) {
                $q->where('id', $packageId);
            });
        }

        // 📅 Purchase Date Range filter
        if ($request->filled('purchase_date_range')) {
            [$start, $end] = explode(' to ', $request->purchase_date_range);
            $query->whereHas('company', function ($q) use ($start, $end) {
                $q->whereBetween('purchase_date', [$start, $end]);
            });
        }

        // 📅 Expire Date Range filter
        if ($request->filled('expire_date_range')) {
            [$start, $end] = explode(' to ', $request->expire_date_range);
            $query->whereHas('company', function ($q) use ($start, $end) {
                $q->whereBetween('expire_date', [$start, $end]);
            });
        }


        $payments = $query->orderBy('id', 'desc')->paginate(20);

        $filters = $request->only(['search', 'status', 'package_id']);
        if (!empty($filters)) {
            $payments->appends($filters);
        }

        $paymentData = Payment_history::select(
        DB::raw('DATE(created_at) as date'),
        DB::raw('SUM(amount) as total_amount')
        )
        ->groupBy('date')
        ->orderBy('date', 'ASC')
        ->get();

        // Chart.js এর জন্য label আর data আলাদা করে পাঠানো
        $labels = $paymentData->pluck('date');
        $data   = $paymentData->pluck('total_amount');

        return view('admin.payment_history.index', compact('payments', 'packages', 'labels', 'data'));
    }








}
