<?php

namespace App\Http\Controllers\client;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Package;
use App\Models\Payment_history;
use App\Models\Company;
use App\Models\FileUploader;
use Carbon\Carbon;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\DB;


class PackageController extends Controller
{
    public function index(Request $request)
    {
        $query = Package::where('status', 1);

        if ($request->has('search') && $request->search != '') {
            $query->where('title', 'LIKE', '%' . $request->search . '%');
        }

        $packages = $query->orderBy('id', 'asc')->paginate(20);

        if ($request->has('search')) {
            $packages->appends(['search' => $request->search]);
        }

        $purchasedPackageIds = Company::where('user_id', auth()->id())
                            ->where('status', 'active')
                            ->first();

        return view('client.package.index', compact('packages', 'purchasedPackageIds'));
    }

    public function purchasePackage($id)
    {
        $page_data['package'] = Package::findOrFail($id);

        return view('client.package.purchase', $page_data);
    }

    

   public function checkName(Request $request)
    {
        $name = trim($request->name);

        $exists = Company::where('name', $name)->exists();

        if ($exists) {
            return response()->json([
                'status' => 'taken',
                'message' => 'The name is already taken.'
            ]);
        } else {
            $baseDomain = get_settings('domain'); 
            
            $parsedUrl = parse_url($baseDomain, PHP_URL_HOST); 
            if (!$parsedUrl) {
                $parsedUrl = str_replace(['http://', 'https://'], '', $baseDomain);
            }

            $subdomain = strtolower(preg_replace('/\s+/', '', $name));

            $url = 'https://' . $subdomain . '.' . $parsedUrl . '/';

            return response()->json([
                'status' => 'available',
                'url' => $url
            ]);
        }
    }


    public function payout(Request $request, $id)
    {
        // Validate incoming request
        $request->validate([
            'company_name'          => 'required|string|max:255', // Client must provide domain
        ]);

        // Fetch the selected package
        $package = Package::find($id);


        if (!$package) {
            return redirect()->back()->with('error', 'Package not found.');
        }

        // Build item array
        $items = [[
            'id'             => $package->id,
            'title'          => $package->title,
            'price'          => $package->price,
        ]];

                
        // Prepare payment details for session
        $payment_details = [
            'items' => $items,

            'custom_field' => [
                'item_type'       => 'package',
                'pay_for'         => 'package purchase',
                'user_id'         => auth()->id(),
                'user_photo'      => auth()->user()->photo ?? '',
                'package_id'      => $package->id,
                'company_name'    => $request->company_name,
                'company_url'     => $request->company_url,
            ],

            'success_method' => [
                'model_name'    => 'PurchasePackage',
                'function_name' => 'purchase_package', // you will handle in success controller
            ],

            'payable_amount' => $package->price,
            'cancel_url'     => route('client.package'),    // Redirect if canceled
            // 'success_url'    => route('payment.success', ''), // Will finalize purchase
            'success_url'    => route('client.payment.success', ''),
        ];

        // Store payment details in session
        Session::put('payment_details', $payment_details);

        // Redirect to payment gateway page
        return redirect()->route('payment');
    }


    public function payment_success(Request $request, $identifier = "")
    {

        $payment_details = session('payment_details');

        $payment_gateway = DB::table('payment_gateways')->where('identifier', $identifier)->first();

        
        $model_name      = $payment_gateway->model_name;
        $model_full_path = str_replace(' ', '', 'App\Models\payment_gateway\ ' . $model_name);

        $status = $model_full_path::payment_status($identifier, $request->all());
            if ($status === true) {
                // ----- INSERT INTO PAYMENT HISTORY -----
                $user_id    = $payment_details['custom_field']['user_id'];
                $package_id = $payment_details['custom_field']['package_id'];
                $amount     = $payment_details['payable_amount'];
                $transaction_id = $request->input('transaction_id', uniqid('txn_')); 
                $session_id = session()->getId();

                $package = Package::find($package_id);
                $purchase_date = Carbon::now();
                $interval = strtolower(trim($package->interval));
                
                // calculate expire date based on package interval
                switch ($package->interval) {
                    case 'day':
                        $expire_date = $purchase_date->copy()->addDays($package->interval_preiod);
                        break;

                    case 'monthly':
                        $expire_date = $purchase_date->copy()->addMonths($package->interval_preiod);
                        break;

                    case 'yearly':
                        $expire_date = $purchase_date->copy()->addYears($package->interval_preiod);
                        break;

                    default:
                        $expire_date = $purchase_date->copy()->addMonths(12); // fallback
                        break;
                }

                $company = Company::create([
                    'user_id'       => $user_id,
                    'name'          => $payment_details['custom_field']['company_name'],
                    'company_url'   => $payment_details['custom_field']['company_url'],
                    'status'        => 'active',
                    'slug'          => slugify($payment_details['custom_field']['company_name']),
                    'package_id'    => $package_id,
                    'purchase_date' => $purchase_date,
                    'expire_date'   => $expire_date,
                ]);

                $db_name = app(\App\Http\Controllers\CommonController::class)
                ->create_organization_db($company->name, auth()->user());

                $company->update([
                    'db_name' => $db_name
                ]);

                Payment_history::create([
                    'user_id'        => $user_id,
                    'company_id'     => $company->id,
                    'payment_type'   => $identifier,
                    'package_id'     => $package_id,
                    'amount'         => $amount,
                    'invoice'        => strtoupper(uniqid('INV_')),
                    'date_added'     => Carbon::now(),
                    'last_modified'  => Carbon::now(),
                    'transaction_id' => $transaction_id,
                    'session_id'     => $session_id,
                    'created_at'     => Carbon::now(),
                    'updated_at'     => Carbon::now(),
                ]);


                Session::flash('success', get_phrase('Payment successful! Package activated.'));
                return redirect()->route('client.package');

        } elseif ($status == "submitted") {
            Session::flash('success', get_phrase('Your payment submitted. It will take some times to enrol.'));
            return redirect(route('home'));
        } else {
            Session::flash('error', get_phrase('Payment failed! Please try again.'));
            redirect()->to($payment_details['cancel_url']);
        }

    }

    public function subscriptionHistory(Request $request)
    {
        $query = Payment_history::with(['company', 'package'])
            ->where('user_id', auth()->user()->id);

        if ($request->has('search') && $request->search != '') {
            $search = $request->search;

            $query->whereHas('company', function ($q) use ($search) {
                $q->where('name', 'LIKE', '%' . $search . '%');
            })->orWhereHas('package', function ($q) use ($search) {
                $q->where('title', 'LIKE', '%' . $search . '%');
            });
        }

        $payments = $query->orderBy('id', 'desc')->paginate(20);

        if ($request->has('search')) {
            $payments->appends(['search' => $request->search]);
        }

        $activeSubscription = Payment_history::with(['company', 'package'])
        ->where('user_id', auth()->user()->id)
        ->whereHas('company', function ($query) {
            $query->whereIn('status', ['active','expired','canceled']);
        })
        ->orderBy('id', 'desc')
        ->first();

        return view('client.subscription_history.index', compact('payments', 'activeSubscription'));
    }

    public function CancelSubscription($id)
    {
        $company = Company::findOrFail($id);

         $company->update([
            'status'           => 'canceled',
        ]);

        return redirect()->route('client.package')
            ->with('success', get_phrase('Subscription canceled successfully'));
    }


}
